/*
 * timer_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vHWTimerTask task configures the Timer 0 peripheral of the TM4C1294NCPDT
 * MCU to run in split pair mode with Timer A and B both configured for
 * periodic up-count edge capture.  The capture event for Timer A is set to be
 * a rising / positive edge trigger on pin PL4.  The capture event for Timer B
 * is set to be a falling / negative edge trigger on pin PL5.  The timer is
 * loaded with the maximum value of 24-bits which means using both the timer
 * load register as well as the prescaler.
 *
 * After the timer is enabled, it will monitor the PL4 and PL5 pins to capture
 * any rising or falling edge events.  An interrupt will fire on either event.
 * When a rising edge event occurs, the ISR stores the current timer values as
 * the start time.  Once a falling edge event occurs, the current timer values
 * are stored again and then the processing is deferred to prvTimerIntTask.
 * 
 * The prvTimerIntTask checks for any potential overflow before calculating the
 * delta in the two time measurements and translate that into an output that is
 * based on microseconds to indicate the duration of the high pulse.  The final
 * result is then sent over UART to display on a terminal window.  The same
 * logic can be applied to also measure a low period.
 *
 * There are some limitations with this method to be aware of.  The limitation
 * on the lower end is if a signal is too fast to sample with the timers due to
 * the time to execute code.  The limitation on the upper end is if the signal
 * period is longer than 139.8 milliseconds at which point the 24-bit timer
 * would overflow.  That issue can be worked around by using two 32-bit timers
 * which would allow measuring a signal up to 35.79 seconds long.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_timer.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/timer.h"
#include "driverlib/uart.h"
#include "utils/uartstdio.h"

/*-----------------------------------------------------------*/

/*
 * Variables to store Timer values from the ISR, must be volatile.
 */
volatile uint32_t g_ui32HighStartCount;
volatile uint32_t g_ui32HighEndCount;

/*
 * Declare a variable that is used to hold the handle of the timer
 * interrupt task.
 */
TaskHandle_t xTimerIntTask = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvTimerIntTask( void *pvParameters );

/*
 * Called by main() to create the hardware timer task.
 */
void vHWTimerTask( void );

/* 
 * Configure the hardware timer 0A on the TM4C1294NCPDT to run in capture-
 * compare mode with interrupts for both rising and falling edges.
*/
static void prvConfigureHWTimer( void );
/*-----------------------------------------------------------*/

void vHWTimerTask( void )
{
    /* Configure the hardware timer to run in capture-compare mode. */
    prvConfigureHWTimer();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Timer 0B - for debug only as it is not used by
     *    the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTimerIntTask,
                 "Timer0B",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 &xTimerIntTask );

    /* Print the initial instructions. */
    UARTprintf("Edge Capture Example for measuring the high period of an"
               "input signal.\n");
    UARTprintf("Input a square wave onto pins PL4 and PL5 of the "
               "EK-TM4C1294XL.\n");

    /* Tasks have been created, now enable Timers 0A and 0B.
     * Depending on the application, this can be placed inside of the Configure
     * Hardware function too.  For this example, the task to receive the ISR
     * notification needs to be created first, so the Timers are only enabled
     * after the task has been created. */
    TimerEnable(TIMER0_BASE, TIMER_BOTH);
}
/*-----------------------------------------------------------*/

static void prvTimerIntTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t ui32HighPeriod = 0;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
        interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* Simple check to avoid overflow cases.  The End Count is the
            second measurement taken and therefore should never be smaller
            than the Start Count unless the timer has overflowed.  If that
            occurs, then add 2^24-1 to the End Count before subtraction. */
            if (g_ui32HighEndCount > g_ui32HighStartCount)
            {
                ui32HighPeriod = g_ui32HighEndCount - g_ui32HighStartCount;
            }
            else
            {
                ui32HighPeriod =
                        (g_ui32HighEndCount + 16777215) - g_ui32HighStartCount;
            }

            /* The System Clock speed is needed to calculate the period length
            of the input signal.  Dividing by the system clock gives the
            period in seconds.  Multiplying by 10^6 then shifts the result
            to microseconds to simplify the UART output of the result. */
            ui32HighPeriod /= (configCPU_CLOCK_HZ / 1000000);

            /* Output the result of the measured input signal high period. */
            UARTprintf("Input Signal Period = %d microseconds\n",
                       ui32HighPeriod);
        }
        else
        {
            /* Timed out. */
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureHWTimer( void )
{
    /* The Timer 0 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER0);

    /* Configure PL4 and PL5 as the CCP pins for Timer 0. */
    GPIOPinTypeTimer(GPIO_PORTL_BASE, GPIO_PIN_4 | GPIO_PIN_5);
    GPIOPinConfigure(GPIO_PL4_T0CCP0);
    GPIOPinConfigure(GPIO_PL5_T0CCP1);

    /* Initialize Timers A and B to both run as periodic up-count edge capture
    timers.  This will split the 32-bit timer into two 16-bit timers. */
    TimerConfigure(TIMER0_BASE, (TIMER_CFG_SPLIT_PAIR |
                                 TIMER_CFG_A_PERIODIC |
                                 TIMER_CFG_A_CAP_TIME_UP |
                                 TIMER_CFG_B_PERIODIC |
                                 TIMER_CFG_B_CAP_TIME_UP));

    /*  To use the timer in Edge Time mode, it must be preloaded with initial
    values.  If the prescaler is used, then it must be preloaded as well.
    Since we want to use all 24-bits for both timers it will be loaded with the
    maximum of 0xFFFF for the 16-bit wide split timers, and 0xFF to add the
    additional 8-bits to the split timers with the prescaler. */
    TimerLoadSet(TIMER0_BASE, TIMER_BOTH, 0xFFFF);
    TimerPrescaleSet(TIMER0_BASE, TIMER_BOTH, 0xFF);

    /* Configure Timer A to trigger on a Positive Edge and configure Timer B
    to trigger on a Negative Edge. */
    TimerControlEvent(TIMER0_BASE, TIMER_A, TIMER_EVENT_POS_EDGE);
    TimerControlEvent(TIMER0_BASE, TIMER_B, TIMER_EVENT_NEG_EDGE);

    /* Enable the Timer A and B interrupts for Capture Events. */
    TimerIntEnable(TIMER0_BASE, TIMER_CAPA_EVENT | TIMER_CAPB_EVENT);

    /* Enable the Timer 0A and 0B interrupts in the NVIC. */
    IntEnable(INT_TIMER0A);
    IntEnable(INT_TIMER0B);
}
/*-----------------------------------------------------------*/

void xTimer0AHandler( void )
{
    /* This ISR is only to store the initial data, no processing required. */

    /* Clear the interrupt. */
    TimerIntClear(TIMER0_BASE, TIMER_CAPA_EVENT);

    /* Store the start time.  In Edge Time Mode, the prescaler is used for the
    the most significant bits.  Therefore, it must be shifted by 16 before
    being added onto the final value. */
    g_ui32HighStartCount = (TimerValueGet(TIMER0_BASE, TIMER_A)) +
                           (TimerPrescaleGet(TIMER0_BASE, TIMER_A) << 16);
}
/*-----------------------------------------------------------*/

void xTimer0BHandler( void )
{
BaseType_t xHigherPriorityTaskWoken;

    /* Clear the interrupt. */
    TimerIntClear(TIMER0_BASE, TIMER_CAPB_EVENT);

    /* Store the end time.  In Edge Time Mode, the prescaler is used for the
    the most significant bits.  Therefore, it must be shifted by 16 before
    being added onto the final value. */
    g_ui32HighEndCount = (TimerValueGet(TIMER0_BASE, TIMER_B)) +
                         (TimerPrescaleGet(TIMER0_BASE, TIMER_B) << 16);

    /* The xHigherPriorityTaskWoken parameter must be initialized to pdFALSE as
    it will get set to pdTRUE inside the interrupt safe API function if a
    context switch is required. */
    xHigherPriorityTaskWoken = pdFALSE;

    /* Send a notification directly to the task to which interrupt processing
    is being deferred. */
    vTaskNotifyGiveFromISR( xTimerIntTask, &xHigherPriorityTaskWoken );

    /* This FreeRTOS API call will handle the context switch if it is required
    or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xHigherPriorityTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}


